#!/usr/bin/perl -w
#
#	uk-muxes.pl	: Generate transmitter mux data from 700-plan-clearance.xslx
#			  from https://www.ofcom.org.uk/siteassets/resources/documents/spectrum/tv-transmitter-guidance/700-plan-clearance.xlsx
#	Deri James	: 9 Dec 2025
#
# Params:-
#
# 	$1 = file location
#

use strict;
use Spreadsheet::ParseExcel;

my $Modes=
{
    'PSB1'	=> ["DVBT", "8000000", "2/3", "NONE", "QAM/64", "8K", "1/32", "NONE", "AUTO"],
    'PSB2'	=> ["DVBT", "8000000", "2/3", "NONE", "QAM/64", "8K", "1/32", "NONE", "AUTO"],
    'PSB3'	=> ["DVBT2", "8000000", "2/3", "NONE", "QAM/256", "32K", "1/128", "NONE", "AUTO"],
    'COM4'	=> ["DVBT", "8000000", "3/4", "NONE", "QAM/64", "8K", "1/32", "NONE", "AUTO"],
    'COM5'	=> ["DVBT", "8000000", "3/4", "NONE", "QAM/64", "8K", "1/32", "NONE", "AUTO"],
    'COM6'	=> ["DVBT", "8000000", "3/4", "NONE", "QAM/64", "8K", "1/32", "NONE", "AUTO"],
    'LTVMux'	=> ["DVBT", "8000000", "3/4", "NONE", "QPSK", "8K", "1/32", "NONE", "AUTO"],
    'NIMux'	=> ["DVBT2", "8000000", "2/3", "NONE", "QPSK", "32K", "1/32", "NONE", "AUTO"],
    'GIMux'	=> ["DVBT", "8000000", "3/4", "NONE", "QAM/16", "8K", "1/32", "NONE", "AUTO"],
};

my $sites={};
my $sitenotes={};
my $today=`date`;
chomp $today;

use Text::Iconv;
my $converter = Text::Iconv->new("utf-8", "windows-1251");
# Text::Iconv is not really required.
# This can be any object with the convert method. Or nothing.
use Spreadsheet::XLSX;
my $fn=shift;
my $excel = Spreadsheet::XLSX->new($fn, $converter) or die "Excel file not recognised";
my $sheet=$excel->{Worksheet}[0];
die "PSB_COM_Muxes not present" if $sheet->{Name} ne "PSB_COM_Muxes";
die "Format not recognised " if $sheet->{Cells}[0][17]->{Val} ne 'PSB1 Channel';
my ($min,$max)=$sheet->row_range();

Populate('P',5,35,17,20,23,26,29,32);

$sheet=$excel->{Worksheet}[1];
die "LTV_Muxes not present" if $sheet->{Name} ne "LTVMux_NIMux_GIMux";
die "Format not recognised " if $sheet->{Cells}[0][12]->{Val} ne 'LTVMux Channel';
($min,$max)=$sheet->row_range();

Populate('L',3,26,12,17,22);

Produce();

sub Populate
{
    my $type=shift;
    my $sitenm=shift;
    my $notes=shift;

    foreach my $r (1..$max)
    {
	my $n=$sheet->{Cells}[$r][$notes]->{Val} || '';
	my $site=$sheet->{Cells}[$r][$sitenm]->{Val};
	next if !defined($site);

	if ($site=~m/^([\w ]+?) (\(.*\))/)
	{
	    $site=$1;
	    $n.=" $2";
	}

	if ($type eq 'P')
	{
	    foreach my $j (1,2,3,5,$notes) {push(@{$sitenotes->{$site}},$sheet->{Cells}[$r][$j]->{Val} || '');};
	}

	foreach my $c (@_)
	{
	    my $mux=$sheet->{Cells}[0][$c]->{Val};
	    $mux=~s/ .*$//;
	    my $o=$c+1;
	    $o++ if ($sheet->{Cells}[0][$o]->{Val}!~m/Offset$/);
	    my $n='';
	    $n=$sheet->{Cells}[$r][$notes]->{Val} || '' if $type eq 'L';
	    my $chan=$sheet->{Cells}[$r][$c]->{Val};

	    if ($chan)
	    {
		my $freq=($chan-21)*8000000+474000000;
		my $offs=$sheet->{Cells}[$r][$o]->{Val} || '';
		$freq-=167000 if $offs eq '-';
		$freq+=167000 if $offs eq '+';
		push(@{$sites->{$site}},[$mux,$freq,$chan.$offs,$n]);
	    }
	}
    }

}

sub Produce
{
    foreach my $site (sort keys %{$sitenotes})
    {
	my $fn=$site;
	$fn=~tr[ ][]d;
	open(my $fh,'>',"uk-$fn") or die "$!";
	print $fh <<"EOF";
#---------------------------------------------------------------
# Generated from:
# <https://www.ofcom.org.uk/siteassets/resources/documents/spectrum/tv-transmitter-guidance/700-plan-clearance.xlsx>
# Date: $today
#---------------------------------------------------------------
# ITV Region:   $sitenotes->{$site}[0]
# BBC Region:   $sitenotes->{$site}[1]
# Group:        $sitenotes->{$site}[2]
# Site Name:    $sitenotes->{$site}[3]
# Notes:        $sitenotes->{$site}[4]
#---------------------------------------------------------------
EOF
        foreach my $mux (@{$sites->{$site}})
        {
            print $fh "[C$mux->[2] $mux->[0] $mux->[3]]\n";
            my $muxnm=$mux->[0];
            print $fh <<"EOF";
	DELIVERY_SYSTEM = $Modes->{$muxnm}[0]
	FREQUENCY = $mux->[1]
	BANDWIDTH_HZ = $Modes->{$muxnm}[1]
	CODE_RATE_HP = $Modes->{$muxnm}[2]
	CODE_RATE_LP = $Modes->{$muxnm}[3]
	MODULATION = $Modes->{$muxnm}[4]
	TRANSMISSION_MODE = $Modes->{$muxnm}[5]
	GUARD_INTERVAL = $Modes->{$muxnm}[6]
	HIERARCHY = $Modes->{$muxnm}[7]
	INVERSION = $Modes->{$muxnm}[8]

EOF
        }
        close($fh);
    }
}
print "Fin\n";

my $PCcols=
{
    'PSB1'	=> 18,
    'PSB2'	=> 21,
    'PSB3'	=> 24,
    'COM4'	=> 27,

}
